/* Genuine Channels product.
 * 
 * Copyright (c) 2002-2007 Dmitry Belikov. All rights reserved.
 * 
 * This source code comes under and must be used and distributed according to the Genuine Channels license agreement.
 */

using System;

namespace Belikov.GenuineChannels.Logbook
{
	/// <summary>
	/// Contains log message types (subcategories)
	/// that can be used to mark a similar messages generated by different entities.
	/// </summary>
	public enum LogMessageType
	{
		/// <summary>
		/// Is used to mark warnings.
		/// </summary>
		Warning,

		/// <summary>
		/// Is used to mark errors.
		/// </summary>
		Error,

		/// <summary>
		/// For critical errors.
		/// </summary>
		CriticalError,

		/// <summary>
		/// The channels' event is raised.
		/// </summary>
		ChannelEvent,


		#region -- Message Processing --------------------------------------------------------------

		/// <summary>
		/// For messages being sent to the remote host.
		/// </summary>
		MessageCreated,

		/// <summary>
		/// For received messages.
		/// </summary>
		MessageReceived,

		/// <summary>
		/// The received request is being invoked.
		/// </summary>
		MessageRequestInvoking,

		/// <summary>
		/// The invocation has been completed.
		/// </summary>
		MessageRequestInvoked,

		/// <summary>
		/// Assembled Security Session Parameters.
		/// </summary>
		SecuritySessionParametersAssembled,

		/// <summary>
		/// For messages being sent to the remote host.
		/// </summary>
		SecuritySessionApplied,

		/// <summary>
		/// For messages being sent to the remote host.
		/// </summary>
		SecuritySessionHasNotBeenEstablishedYet,

		/// <summary>
		/// The connection is available and the message is being sent.
		/// </summary>
		MessageIsSentSynchronously,

		/// <summary>
		/// The connection is available and the message is being sent.
		/// </summary>
		MessageIsSentAsynchronously,

		/// <summary>
		/// The connection is available and message is being sent.
		/// </summary>
		MessageIsEnqueued,

		/// <summary>
		/// The message has been dequeued.
		/// </summary>
		MessageHasBeenDequeued,

		/// <summary>
		/// The exception is being dispatched to the caller context.
		/// </summary>
		ExceptionDispatched,

		/// <summary>
		/// The message is being dispatched to the caller context.
		/// </summary>
		MessageDispatched,

		#endregion

		#region -- Security Session Lifetime -------------------------------------------------------

		/// <summary>
		/// A key provider has been associated with a particular name.
		/// </summary>
		KeyProviderAssociated,

		/// <summary>
		/// A key provider has been dissociated from a particular name.
		/// </summary>
		KeyProviderDissociated,

		/// <summary>
		/// Security Session Created.
		/// </summary>
		SecuritySessionCreated,

		/// <summary>
		/// Security Session establishing initiated.
		/// </summary>
		SecuritySessionInitiated,

		/// <summary>
		/// Security Session is being established.
		/// </summary>
		SecuritySessionEstablishing,

		/// <summary>
		/// The security key of the Security Session.
		/// </summary>
		SecuritySessionKey,

		/// <summary>
		/// Security Session is established.
		/// </summary>
		SecuritySessionEstablished,

		/// <summary>
		/// Security Session has been destroyed.
		/// </summary>
		SecuritySessionDestroyed,

		#endregion

		#region -- HostInformation Creation --------------------------------------------------------

		/// <summary>
		/// The HostInformation has been created.
		/// </summary>
		HostInformationCreated,

		/// <summary>
		/// The HostInformation is released.
		/// </summary>
		HostInformationReleased,

		/// <summary>
		/// The URI of the HostInformation is updated.
		/// </summary>
		HostInformationUriUpdated,

		/// <summary>
		/// The association between HostInformation and its URI and URL has been broken.
		/// </summary>
		HostInformationReferencesDisassociated,

		#endregion

		#region -- Connection stuff ----------------------------------------------------------------

		/// <summary>
		/// Listening a particular local end point has been started.
		/// </summary>
		ListeningStarted,

		/// <summary>
		/// A connection is being accepted.
		/// </summary>
		ConnectionAccepting,

		/// <summary>
		/// Listening has been stopped.
		/// </summary>
		ListeningStopped,

		/// <summary>
		/// The list of connection parameters.
		/// </summary>
		ConnectionParameters,

		/// <summary>
		/// A connection is being established.
		/// </summary>
		ConnectionEstablishing,

		/// <summary>
		/// The Connection Level Security Session is established.
		/// </summary>
		ConnectionSecurityIsEstablished,

		/// <summary>
		/// The connection has been established.
		/// </summary>
		ConnectionEstablished,

		/// <summary>
		/// The connection is being reestablishing.
		/// </summary>
		ConnectionReestablishing,

		/// <summary>
		/// The connection is reestablished.
		/// </summary>
		ConnectionReestablished,

		/// <summary>
		/// Stream desynchronization occurred.
		/// </summary>
		ConnectionStreamDesynchronization,

		/// <summary>
		/// Receiving has been initiated.
		/// </summary>
		ReceivingStarted,

		/// <summary>
		/// The message has been received.
		/// </summary>
		ReceivingFinished,

		/// <summary>
		/// Asynchronous sending has been started.
		/// </summary>
		AsynchronousSendingStarted,

		/// <summary>
		/// Asynchronous sending has been completed.
		/// </summary>
		AsynchronousSendingFinished,

		/// <summary>
		/// The message has been sent asynchronously.
		/// </summary>
		MessageHasBeenSentAsynchronously,

		/// <summary>
		/// Synchronous sending has been started.
		/// </summary>
		SynchronousSendingStarted,

		/// <summary>
		/// Synchronous sending has been finished.
		/// </summary>
		SynchronousSendingFinished,

		/// <summary>
		/// The message has been sent synchronously.
		/// </summary>
		MessageHasBeenSentSynchronously,

		/// <summary>
		/// A ping sending has been initiated.
		/// </summary>
		ConnectionPingSending,

		/// <summary>
		/// The ping has been sent.
		/// </summary>
		ConnectionPingSent,

		/// <summary>
		/// The ping has been received.
		/// </summary>
		ConnectionPingReceived,

		/// <summary>
		/// The connection is being released.
		/// </summary>
		ReleaseConnections,

		/// <summary>
		/// The connection is being shutting down.
		/// </summary>
		ConnectionShuttingDown,

		/// <summary>
		/// The connection is (being) closed.
		/// </summary>
		ConnectionFailed,

		#endregion

		#region -- Broadcast Engine ----------------------------------------------------------------

		/// <summary>
		/// The broadcast invocation has been initiated.
		/// </summary>
		BroadcastInvocationInitiated,

		/// <summary>
		/// The broadcast filter is applied.
		/// </summary>
		BroadcastFilterCalled,

		/// <summary>
		/// The broadcast recipient is being invoked.
		/// </summary>
		BroadcastRecipientInvoked,

		/// <summary>
		/// The broadcast recipient is being invoked on the second stage of broadcast.
		/// </summary>
		BroadcastRecipientInvokedAfterTimeout,

		/// <summary>
		/// The response is being parsed.
		/// </summary>
		BroadcastResponseParsed,

		/// <summary>
		/// The broadcast invocation has been finished.
		/// </summary>
		BroadcastInvocationCompleted,

		/// <summary>
		/// The broadcast recipient is added.
		/// </summary>
		BroadcastRecipientAdded,

		/// <summary>
		/// The broadcast recipient is deleted.
		/// </summary>
		BroadcastRecipientRemoved,

		#endregion

		#region -- Debugging Messages --------------------------------------------------------------

		/// <summary>
		/// Warning about the current application state.
		/// </summary>
		DebuggingWarning,

		/// <summary>
		/// Error or a wrong state.
		/// </summary>
		DebuggingError,

		/// <summary>
		/// The try failed.
		/// </summary>
		DebuggingFailed,

		/// <summary>
		/// The try is successful.
		/// </summary>
		DebuggingSuccess,

		#endregion

		#region -- Low-level Transport Messages ----------------------------------------------------

		/// <summary>
		/// Synchronous sending is being initiated.
		/// </summary>
		LowLevelTransport_SyncSendingInitiating,

		/// <summary>
		/// Synchronous sending has been initiated.
		/// </summary>
		LowLevelTransport_SyncSendingInitiated,

		/// <summary>
		/// Synchronous sending has been completed.
		/// </summary>
		LowLevelTransport_SyncSendingCompleted,

		/// <summary>
		/// Synchronous receiving is being initiated.
		/// </summary>
		LowLevelTransport_SyncReceivingInitiating,

		/// <summary>
		/// Synchronous receiving has been initiated.
		/// </summary>
		LowLevelTransport_SyncReceivingInitiated,

		/// <summary>
		/// Synchronous receiving has been completed.
		/// </summary>
		LowLevelTransport_SyncReceivingCompleted,

		/// <summary>
		/// Asynchronous sending is being initiated.
		/// </summary>
		LowLevelTransport_AsyncSendingInitiating,

		/// <summary>
		/// Asynchronous sending has been initiated.
		/// </summary>
		LowLevelTransport_AsyncSendingInitiated,

		/// <summary>
		/// Asynchronous sending has been completed.
		/// </summary>
		LowLevelTransport_AsyncSendingCompleted,

		/// <summary>
		/// Asynchronous receiving is being initiated.
		/// </summary>
		LowLevelTransport_AsyncReceivingInitiating,

		/// <summary>
		/// Asynchronous receiving has been initiated.
		/// </summary>
		LowLevelTransport_AsyncReceivingInitiated,

		/// <summary>
		/// Asynchronous receiving completed.
		/// </summary>
		LowLevelTransport_AsyncReceivingCompleted,

		#endregion

		#region -- Statistic counters --------------------------------------------------------------

		/// <summary>
		/// Statistics of the usage of the thread pool.
		/// </summary>
		ThreadPoolUsage,

		#endregion

		/// <summary>
		/// The connection has been selected.
		/// </summary>
		ConnectionSelected,
	}
}
